/**
 * \file: util_trace.h
 *
 * \version: $Id: util_trace.h,v 1.14 2012/02/15 12:31:37 bkebianyor Exp $
 *
 * Definitions for the trace part of the utility library. 
 *
 * \component: Utility Library
 *
 * \author: T. Polle <tpolle(o)de.adit-jv.com>
 *
 * \copyright: (c) 2003 - 2008 ADIT Corporation
 */
#ifndef __UTIL_TRACE_H_
#define __UTIL_TRACE_H_

#ifdef __cplusplus
extern "C" {
#endif

/* Filesystem operations. */
typedef enum
{
  drv_fs_open     =  1,
  drv_fs_close    =  2,
  drv_fs_read     =  3,
  drv_fs_write    =  4,
  drv_fs_fstat    =  5,
  drv_fs_stat     =  6,
  drv_fs_rename   =  7,
  drv_fs_chmod    =  8,
  drv_fs_fchmod   =  9,
  drv_fs_mkdir    = 10,
  drv_fs_rmdir    = 11,
  drv_fs_unlink   = 12,
  drv_fs_access   = 13,
  drv_fs_readdir  = 14,
  drv_fs_fsync    = 15,
  drv_fs_truncate = 16,
  drv_fs_statvfs  = 17,
  drv_fs_fstatvfs = 18,
  drv_fs_chdir    = 19,
  drv_fs_fchdir   = 20
} drv_fs_op_t;


#  include <stdarg.h>
#  include "adit_typedef.h"
#  include "adit_trace.h"
#  include "trace_interface.h"

/**
 * Definitions for functions with lazy binding.
 */
typedef U32 (*UTIL_trace_out_t)
  (U32 uwLen, U16 trClass, TR_tenTraceLevel trLevel, const U8* p_ubTrData);
typedef BOOL(*UTIL_is_class_selected_t)
  (U16 enClass, TR_tenTraceLevel enLevel);
typedef  BOOL (*UTIL_register_channel_t)
  (TR_tenTraceChan chan_id, const TRACE_CALLBACK p_Func);
typedef BOOL (*UTIL_unregister_channel_t)
  (TR_tenTraceChan chan_id, const TRACE_CALLBACK p_Func);
typedef U32 (*UTIL_TraceBinOutput_t)
  (U32 uwCompId, U32 uwSockId, U32 uwBufferLen, U8* p_ubDataBuffer);

/**
 * \brief Checks whether trace is enable for the trace level.
 * 
 * The field \a fIsActive in \a data is also updated to hold the
 * latest result.
 *
 * No buffer has to be provided.
 *
 * \param data      definition of trace class and trace level
 *
 * \returns \li TRUE,  if the trace level is activated.
 *          \li FALSE, else
 */
IMPORT BOOL UTIL_trace_isActive(TraceData* data);

/**
 * \brief Interface to TTFis where only binary data is passed.
 *
 * No buffer has to be provided.
 *
 * \param data      definition of trace class and trace level
 * \param buf       buffer with binary data
 * \param size      size of the buffer
 *
 * \return none
 */
IMPORT void UTIL_traceb(TraceData* data, U8* buf, S32 size);

/**
 * \brief Interface to TTFis where only binary data is passed.
 *
 * No buffer has to be provided.
 * Before the format string is parsed a quick check is made whether the 
 * corresponding level in the trace class is active.
 *
 * \param data      definition of trace class and trace level
 * \param buf       buffer with binary data
 * \param size      size of the buffer
 *
 * \return none
 */
IMPORT void UTIL_atraceb(TraceData* data, U8* buf, S32 size);

/**
 * \brief Interface to TTFis where only binary data(MST) is passed.
 *
 * No buffer has to be provided.Data is passed to socket clients, ISP header is added.
 * Meant for RBCM socket clients.
 *
 * \param data      definition of Trace Large Binary streams
 * \param buf       buffer with binary data
 * \param size      size of the buffer
 *
 * \return none
 */
IMPORT void UTIL_traceb_mst(TraceStreamData* data, U8* buf, S32 size);


/**
 * \brief Interface to registry callback function to trace
 *
 * \param channel      definition of callback channel ID and a pointer
 *                     to callback function
 *
 * \return T-Kernel Error Code
 */
IMPORT S32 UTIL_trace_chan_reg(LaunchChannel channel);

/**
 * \brief Interface to unregistry callback function to trace
 *
 * \param channel      definition of callback channel ID and a pointer
 *                     to callback function
 *
 * \return T-Kernel Error Code
 */
IMPORT S32 UTIL_trace_chan_unreg(LaunchChannel channel);

#if defined(ADIT_QAC_CHECK)
#  define TR_(level, args) \
    /* PRQA: Lint Message 717: On purpose. */   \
    /*lint -save -e717 */                       \
    do { } while(0)                             \
    /*lint -restore */

#else
#  define TR_va_args_(...) \
  __VA_ARGS__

enum
{
  buf_sz  =  512  /* maximum size of a trace buffer */
};

IMPORT void UTIL_trace_buf(TraceData* data, char* fmt, va_list args);

LOCAL void TR__(TR_tenTraceLevel tr_level, U32 tr_class, char* fmt, ...)
  ATTRIBUTE((format (printf, 3, 4)));
Inline void TR__(TR_tenTraceLevel tr_level, U32 tr_class, char* fmt, ...)
{
  S8        buf[buf_sz] = {0};
  TraceData data        = {buf_sz,tr_level,tr_class,buf,FALSE};

  va_list   args        = {0};

  data.ubLevel      = tr_level;
  data.uhClass      = tr_class;
  data.p_ubBuffer   = buf;

  va_start(args, fmt);
  UTIL_trace_buf(&data, fmt, args);
  va_end(args);
}

#  define TR_(level, args)                \
  TR__((level), TR_va_args_ args)

#endif /* defined(ADIT_QAC_CHECK) */

/**
 * \ingroup utility
 * \defgroup simple_trace Simple Trace
 * The following macros are used for trace message.
 *
 * The parameters passed to these macros are similar to the parameters,
 * which are passed to the function printf(). The first parameter is the
 * format and describes the subsequent parameters.
 *
 * @param class     trace class
 * @param fmt       format definitions
 *                    %XY: X = size in bytes [1,2,4], 
 *                         Y = [du]
 *                           u: unsigned number
 *                           d: signed number
 *                    %Xs:  string with length X in bytes
 *                          X may be obmitted, if the last parameter passed
 *                          is specified.
 * @param ...       parameters as indicated by the format
 *                    eg. fmt="%1u%s" means the first subsequent parameter is
 *                    an unsigned number with width 1 byte and the
 *                    second paramter is a string
 *
 *
 * The macros correspond to trace levels as follows:
 *
 * TR_F:  This level is fixed on and can not be disabled by the
 *        TTFis. The fatal-error-information is logged in the
 *        post-mortem-buffer. Examples for this levels are system
 *        errors, ASSERTS, Wachtdog-messages etc. When a fatal error
 *        occurs, the last n seconds of the Trace-buffer are saved
 *        into the post-mortem-memory (PMM).
 *
 * TR_E:  This level is active for all classes by default but it can
 *        be enabled / disabled by TTFis. In this level all components
 *        logs software errors. During normal use, no trace traffic
 *        should occur. Examples for this level are: not enough memory
 *        available, wrong parameter set, timeouts etc.
 *
 * TR_SM: In this level only essential trace messages are given
 *        out. The processor load caused by Trace should be as low as
 *        possible (very close to the customer version). Only working
 *        results for overall-components are traced out and allow the
 *        identification of the erroneous system component. The
 *        Trace-volume must be minimized to a value that the system is
 *        not obstructed by Trace when all Trace-classes are enabled
 *        in this level. The number of outgoing Bytes in this level
 *        should be only a fraction of the total Trace-bandwidth
 *        (~1/10).
 *
 * TR_S:  This level provides a more detailed output as the Level
 *        2. It facilitates the exacter localisation of the error that
 *        occurs. This level is also used for tracing out warnings.
 *
 * TR_C:  This level should not be enabled through all Trace-classes,
 *        because the output is bigger than the total
 *        Trace-bandwidth. Only several Trace-classes can be grouped
 *        together to sub-components views. This level enables the
 *        detailed analysis of working results of single
 *        components. In error cases, the identification of the flawed
 *        component is possibly.
 *
 * TR_UX: These levels are free for use by the developer and can be
 *        used for internal output.
 */

/**
 * Consult \ref simple_trace.
 */
#define TR_F(args) \
  TR_(TR_LEVEL_FATAL, args)
/**
 * Consult \ref simple_trace.
 */
#define TR_E(args) \
  TR_(TR_LEVEL_ERRORS, args)
/**
 * Consult \ref simple_trace.
 */
#define TR_SM(args) \
  TR_(TR_LEVEL_SYSTEM_MIN, args)
/**
 * Consult \ref simple_trace.
 */
#define TR_S(args) \
  TR_(TR_LEVEL_SYSTEM, args)
/**
 * Consult \ref simple_trace.
 */
#define TR_C(args) \
  TR_(TR_LEVEL_COMPONENT, args)
/**
 * Consult \ref simple_trace.
 */
#define TR_U1(args) \
  TR_(TR_LEVEL_USER_1, args)
/**
 * Consult \ref simple_trace.
 */
#define TR_U2(args) \
  TR_(TR_LEVEL_USER_2, args)
/**
 * Consult \ref simple_trace.
 */
#define TR_U3(args) \
  TR_(TR_LEVEL_USER_3, args)
/**
 * Consult \ref simple_trace.
 */
#define TR_U4(args) \
  TR_(TR_LEVEL_USER_4, args)


#ifdef __cplusplus
};
#endif

#endif /* __UTIL_TRACE_H_ */
